/*==================================================================*/
/* C Routines for the DAS005 Data Acquisition System                */
/* Author Peter Simmonds											*/
/* Date   25th May 1996 											*/
/* Company															*/
/*==================================================================*/
#include <time.h>

#define TIMEDELAY 50

void InitPrinterPort(int LPTNo,int LPTPortAdd,int *error);
void FindPCSpeed();
int ReadADC(int ChanMode,int Channel,int *error);
int ReadDIP(int IONum,int *error);
int ReadAllDips();
void WriteDOP(int State,int IONum,int *error);
void WriteAllDOP(int DOValue,int *error);
void Wait(int j);

int LPTData,LPTStatus,LPTControl;
unsigned char DataReg,StatusReg,ControlReg;
int gSpeed;
float gain,offset;

void main();

void main()
{
int i,j,jl,k,Err;
	clrscr();
	FindPCSpeed();
	gain=1;  /* the gain and offset can be edited to provide more accurate readings */
	offset=0;
	printf("\n Which Printer Port 1 or 2 - ");
	scanf("%d",&i);
	InitPrinterPort(i,0,&Err);
	for (k=1;k<2000;k++) {
		clrscr();
		printf("***** Read Single Ended ADC Channels *****\n");
		for(i = 1;i<9;i++) { /* Testing the MAX186 ADC */
			j = ReadADC(1, i,&Err);
			j=(int)((float)j*gain)+offset;
			printf("ADC %d=%dmV\n", i, j);
			}
		printf("***** Read Differential ADC Channels *****\n");
		for(i = 1;i<5;i++) {
			j = ReadADC(1, i*2-1,&Err);
			j=(int)((float)j*gain)+offset;
			jl=ReadADC(1,i*2,&Err);
			jl=(int)((float)jl*gain)+offset;
			printf("ADC %d=%dmV\n", i, j-jl);
			}
		/* Reading all the DIPS */
		printf("***** Read All DIPs ******\n");
		i = ReadAllDips();
		printf("Reading all DIPs=%d\n", i);

		/* Reading each of the Single DIPs */
		printf("***** Read Each DIPs ******\n");
		for (i = 1;i<5;i++) {
			j = ReadDIP(i,&Err);
			printf("DIP %d=%d  ", i,j);
			}
		printf("\n");

		/* Writing Single DOPs */
		printf("***** Writing Single DOPs *****\n");
		for( i = 1;i<5;i++) {
			WriteDOP(1, i,&Err); /* turn it on */
			delay(TIMEDELAY);
			WriteDOP(0, i,&Err); /* turn it off */
			delay(TIMEDELAY);
			}

		/* Write to all the DOPs */
		printf("***** Writing to All DOPS *****\n");
		WriteAllDOP(15,&Err);
		printf("Turn all DOPs On\n");
		delay(1000);
		WriteAllDOP(0,&Err);
		printf("Turn all DOPs Off\n");
		delay(1000);
		}
}

/*===================================================================*/
/* A routine to determine the speed of the PC so that delays can be
   calculated for reading the ADC
*/
void FindPCSpeed()
{
clock_t start,end;
float diff,timedelay;
long int i,j;
	printf("Calculating Speed of PC\n");
	j=1000000;
	do {
		start=clock();
		for (i=0;i<j;i++) ;
		end=clock();
		diff=(float)((end-start)/CLK_TCK);
		if (diff<0.2)j=j*10;
		} while(diff<0.2);
	j=(long int)((float)j*2.0/diff);
	start=clock();
	for (i=0;i<j;i++) ;
	end=clock();
	diff=(float)((end-start)/CLK_TCK);
	timedelay=diff/(float)i;
	if (timedelay<0.000002) {
		gSpeed=0.000002/timedelay+1;
		}
	else {
		gSpeed=1;
		}
	printf("\n PC's timedelay=%10.8fuSec gspeed=%d\n",timedelay,gSpeed);
	delay(2000);
}

/*===================================================================*/
/* The ReadADC function reads the MAX186 ADC
 * If ChanMode=0 then it is a differential reading and
 *    Channel can be 1 to 4.
 *			Channel 1=ADC inputs 1 & 2
 *			Channel 2=ADC inputs 3 & 4
 *			Channel 3=ADC inputs 5 & 6
 *			Channel 4=ADC inputs 7 & 8
 * If ChanMode=1 then it is a single ended reading and
 *	  Channel can be 1 to 8 where the ADC input is the same as the
 *	  Channel number.
 * If ChanMode or Channel is outside these ranges *error is returned as
 *    a non zero value.
 */
int ReadADC(int ChanMode,int Channel,int *error)
{
int i,Word,Mask;
char Bit;
	*error=0;
	ControlReg&=0xfd;
	outportb(LPTControl,ControlReg);     /* CS High */
	outportb(LPTData,0x0);        /* SCLK Low, DIN Low */
	switch(ChanMode) {
		case 1:	if(Channel<0 || Channel>8) *error=1;
				break;
		case 0: if(Channel<0 || Channel>4) *error=1;
				break;
		default: *error=1;
				break;
		}
	Channel--;
	Word=0;
	if (ChanMode==1) { /* Single Ended */
		if (Channel & 0x01) Word=Word|0x04;
		if (Channel & 0x02) Word=Word|0x01;
		if (Channel & 0x04) Word=Word|0x02;
		}
	else {
		Word=Channel;
		}
	Word=Word<<4;
	ChanMode<<=2;
	Word=Word | ChanMode;
	Word=Word | 0x8a;

	ControlReg=ControlReg | 0x02;
	outportb(LPTData,0x0);        /* SCLK Low, DIN Low */
	Wait(gSpeed*10);
	outportb(LPTControl,ControlReg);     /* Bring CS Low */
	Wait(gSpeed*10);
	Mask=0x80;
	for(i=0;i<8;i++) {          /* send the control word */
		Bit=Word & Mask;
		if(Bit) {
			outportb(LPTData,0x02);   /* CS Low SCLK Low, DIN High */
			Wait(gSpeed);
			outportb(LPTData,0x03);   /* CS Low SCLK High, DIN High */
			Wait(gSpeed);
			}
		else {
			outportb(LPTData,0x0);    /* CS Low SCLK Low, DIN Low */
			Wait(gSpeed);
			outportb(LPTData,0x01);   /* CS Low SCLK High, DIN Low */
			Wait(gSpeed);
			}
		Mask>>=1;
		}
	outportb(LPTData,0x0);    /* CS Low SCLK Low, DIN Low */
	Wait(20*gSpeed);
	Word=0;
	for(i=0;i<12;i++) {
		Word<<=1;
		outportb(LPTData,0x01);		/* Bring Clk High */
		Wait(gSpeed);
		outportb(LPTData,0x0);		/* Bring Clk Low */
		Wait(gSpeed);
		Bit=inportb(LPTStatus);        /* read ADC's Data Out */
		Bit &=0x08;
		if(Bit) Word|=0x01;
		}
	ControlReg=ControlReg & 0xfd;
	outportb(LPTControl,ControlReg);         /* Bring CS High */
	return(Word);
}

/*===================================================================*/
void Wait(int j)
{
int i,k;
	for(i=0;i<j;i++) k=k+1;
}

/*===================================================================*/
/* The ReadAllDips function reads the four digital inputs 1-4
 * and returns then as an integer with a value of 0 to 15
 * depending on the state of the inputs
 */
int ReadAllDips()
{
unsigned char i,ip;
	outportb(LPTData,DataReg); /* Make sure Digital Outputs don't change */
	ControlReg &= 0xfe;   /* bring strobe low then high to latch the inputs */
	outportb(LPTControl,ControlReg);
	Wait(10*gSpeed);
	ControlReg |= 0x01;
	outportb(LPTControl,ControlReg);
	Wait(10*gSpeed);
	i=inportb(LPTStatus);
	ip=i ^ 0x80;	/* invert the busy signal */
	ip=ip & 0xf0;
	ip>>=4;
	return(ip);
}

/*===================================================================*/
/* The ReadDip function reads the state of a single digital input
 * The number of the input is in "IONum" and can have a value of 1 to 4
 * If IONum is not in the range of 1 to 4 then *error is set to non zero
 */
int ReadDIP(int IONum,int *error)
{
unsigned char i,ip,Mask;
int Bit;
	*error=0;
	if(IONum<1 || IONum>4) *error=1;
	if(!(*error)) {
		outportb(LPTData,DataReg); /* Make sure Digital Outputs don't change */
		ControlReg &= 0xfe;   /* bring strobe low then high to latch the inputs */
		outportb(LPTControl,ControlReg);
		Wait(10*gSpeed);
		ControlReg |= 0x01;
		outportb(LPTControl,ControlReg);
		Wait(10*gSpeed);
		i=inportb(LPTStatus);
		ip=i ^ 0x80;	/* invert the busy signal */
		ip=ip & 0xf0;
		Mask=0x08;
		Mask<<=IONum;
		ip=ip & Mask;
		Bit=0;
		if(ip) Bit=1;
		}
	return(Bit);
}

/*===================================================================*/
/* The WriteDOP function sets or resets a single output "IONum"
 * according to the value in "State".
 * IONum can have the values 1 to 4
 * State can be 0 or 1.
 * If the value of State or IONum are out of the above ranges
 * "*error" is set to non-zero.
 */
void WriteDOP(int State,int IONum,int *error)
{
unsigned char Mask;
	*error=0;
	if(State!=0 && State!=1) *error=1;
	if(IONum<1 || IONum>4) *error=1;
	IONum--;
	if(!(*error)) {
		Mask=0x01;
		Mask<<=IONum;
		if(!State) {
			Mask=~Mask;
			DataReg=DataReg & Mask;
			}
		else DataReg=DataReg | Mask;
		outportb(LPTData,DataReg);
		ControlReg &= 0xfe;   /* bring strobe low then high */
		outportb(LPTControl,ControlReg);
		Wait(10*gSpeed);
		ControlReg |= 0x01;
		outportb(LPTControl,ControlReg);
		Wait(10*gSpeed);
		}
}

/*===================================================================*/
/* The WriteAllDOP function sets or resets the four digital outputs
 * according to the value of "DOValue"
 * DOValue can have the values 0 to 15
 * If the value of DOValue is out of range
 * "*error" is set to non-zero.
 */
void WriteAllDOP(int DOValue,int *error)
{
	*error=0;
	if(DOValue<0 || DOValue>15) *error=1;
	if(!(*error)) {
		DataReg=DOValue;
		outportb(LPTData,DataReg);
		ControlReg &= 0xfe;   /* bring strobe low then high */
		outportb(LPTControl,ControlReg);
		Wait(10*gSpeed);
		ControlReg |= 0x01;
		outportb(LPTControl,ControlReg);
		Wait(10*gSpeed);
		}
}

/*===================================================================*/
/* The function InitPrinterPort initialises the printer port used to control
 * the DAS005.
 * "LPTNo" is the number of the printer port 1 to 3
 * "LPTPortAdd" if it is non zero the LPTNo is ignored and the value of
 * LPTPortAdd is used as the base address
 * If "LPTNo" is out of range "*error" is set to non-zero.
 */
void InitPrinterPort(int LPTNo,int LPTPortAdd,int *error)
{
	*error=0;
	if(!LPTPortAdd)
		switch(LPTNo) {
			case 1: LPTData=0x378;
					break;
			case 2: LPTData=0x278;
					break;
			default: *error=1;
					break;
			}
	else LPTData=LPTPortAdd;
	if(!(*error)) {
		LPTStatus=LPTData+1;
		LPTControl=LPTData+2;
		DataReg=0;
		WriteAllDOP(0,error);
		ControlReg=0x0b;         /* Strobe low Autofd high */
		outportb(LPTControl,ControlReg);
		}
}
